#!/bin/bash

DOCKER_NETWORK="phison-network"
BASE_DIR=$(dirname "$(readlink -f $0)")
if [[ $BASE_DIR == *[' ''!''?'@#\$%^\&*()+]* ]]; then
    echo -e "$(date +%F\ %T) : Folder directory has special character, please do not used special character to naming folder"
    exit 1
fi

#### Execute only root user
if [ "$(id -u)" != "0" ]; then
    msg1="Please execute this script with root privileges."
    echo -e "$(date +%F\ %T) : [error] $msg1"
    exit 1
fi

#### Create folder
DATE=$(date +'%Y-%m-%d_%H%M')
DEST_DIR=$BASE_DIR/Pro_Suite_export_$DATE

if [[ ! -d $DEST_DIR ]]; then
    mkdir -p $DEST_DIR
fi

#### OS and Board
echo -e "$(date +%F\ %T) : Get OS version and board info."
echo "" >> $DEST_DIR/os_board_info.txt
echo "## OS Release" >> $DEST_DIR/os_board_info.txt
cat /etc/os-release >> $DEST_DIR/os_board_info.txt

echo "" >> $DEST_DIR/os_board_info.txt
echo "## Board Info" >> $DEST_DIR/os_board_info.txt
dmidecode -t 2 >> $DEST_DIR/os_board_info.txt

#### PCI
echo -e "$(date +%F\ %T) : Get PCI, CPU, Memory, Disk info."
lspci > $DEST_DIR/pci_info.txt
lscpu > $DEST_DIR/cpu_info.txt
lsmem > $DEST_DIR/mem_info.txt
lsblk > $DEST_DIR/disk_info.txt

#### Nvidia
echo -e "$(date +%F\ %T) : Get display, Nvidia info."
lshw -c display >> $DEST_DIR/display_info.txt
if command -v nvidia-smi &> /dev/null; then
    echo "" >> $DEST_DIR/display_info.txt
    nvidia-smi >> $DEST_DIR/display_info.txt
fi

echo "" >> $DEST_DIR/display_info.txt
dpkg -l | grep nvidia >> $DEST_DIR/display_info.txt

#### Pro Suite config file
echo -e "$(date +%F\ %T) : Get Pro Suite version info."
cp $BASE_DIR/phison-config.env $DEST_DIR/

#### Container logs
if command -v docker &> /dev/null; then
    echo -e "$(date +%F\ %T) : Get Pro Suite docker ps -a."
    docker ps -a >> "$DEST_DIR"/docker_ps.txt 2>&1

    echo -e "$(date +%F\ %T) : Get Pro Suite docker images."
    docker images >> "$DEST_DIR"/docker_images.txt

    docker network ls | grep -q $DOCKER_NETWORK
    result=$?
    if [[ "$result" -eq 0 ]]; then
        echo -e "$(date +%F\ %T) : Get Pro Suite docker container logs by phison-network."
        container_names=($(docker network inspect $DOCKER_NETWORK | jq '.[0].Containers | to_entries | .[] | .value.Name' | tr -d '"'))
        for container in "${container_names[@]}"; do
            docker logs --since 1h "$container" > "$DEST_DIR"/container_"$container".log 2>&1
        done
    fi
fi

#### Install log
echo -e "$(date +%F\ %T) : Get Pro Suite install logs."
LOG_FILE="/var/log/prosuite_install.log"
if [[ -f "$LOG_FILE" ]];then
    cp "$LOG_FILE" "$DEST_DIR"
fi

echo -e "$(date +%F\ %T) : Get Pro Suite service logs."
LOG_DIR="/mnt/nvme0/.phison_log"
if [[ -d "$LOG_DIR" ]];then
    cp -a "$LOG_DIR/." "$DEST_DIR/phison_log/"
fi

LOG_DIR="/opt/phisonai/data/users"
for user_dir in $LOG_DIR/*; do
    USER=$(basename "$user_dir")
    if [[ ! -d "$DEST_DIR/$USER" ]];then
        mkdir $DEST_DIR/$USER
    fi

    if [[ ! -d "$user_dir/jobs" ]];then
        continue
    fi

    for dir in $user_dir/jobs/*; do
        UUID=$(basename "$dir")
        if [[ ! -d "$DEST_DIR/$USER/$UUID" ]];then
            mkdir "$DEST_DIR/$USER/$UUID"
        fi

        cp $dir/*log "$DEST_DIR/$USER/$UUID"
    done
done

#### tar cvf
echo -e "$(date +%F\ %T) : Compress $DEST_DIR to $BASE_DIR/export_hw_$DATE.tar"
tar cvf export_hw_$DATE.tar -C $BASE_DIR Pro_Suite_export_$DATE 2>&1 >/dev/null
result=$?
if [[ "$result" -eq 0 ]]; then
    if [[ -d $DEST_DIR ]]; then
        rm -rf $DEST_DIR
    fi
fi

echo -e "$(date +%F\ %T) : Finish gererate file export_hw_$DATE.tar at $BASE_DIR."