#!/bin/bash

#### LOG File
LOG_FILE="/var/log/prosuite_install.log"

#### Get Current Path, do not contains special charactor
PROSUITE_INSTALL_DIR=$(dirname "$(readlink -f $0)")
if [[ $PROSUITE_INSTALL_DIR == *[' ''!''?'@#\$%^\&*()+]* ]]; then
    echo -e "$(date +%F\ %T) : Folder directory has special character, please do not used special character to naming folder"
    exit 1
fi

#### Pro Suite Data Path
PROSUITE_DATA_DIR="/mnt/data/local-data"
DEFAULT_MODEL_DST="/usr/local/models"

#### Normal install
#### Docker User / Token
PROSUITE_IMAGE_USER="licensesp"
PROSUITE_IMAGE_TOKEN="dckr_pat_2nHaP0kuEFLb4z8iyBcGGeqn0rU"

#### Normal install
#### Huggingface User / Token
HUGGINGFACE_USER="s3171103"
HUGGINGFACE_TOKEN="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"

#### Docker configure
DOCKER_NETWORK="phison-network"
DOCKER_COMPOSE_FILE="$PROSUITE_INSTALL_DIR/phison-compose.yaml"
DOCKER_CONFIG_ENV="$PROSUITE_INSTALL_DIR/phison-config.env"
DOCKER_IMAGE_ENV="$PROSUITE_INSTALL_DIR/phison-image.env"
DOCKER_CLI="docker compose --env-file $DOCKER_CONFIG_ENV --env-file $DOCKER_IMAGE_ENV --file $DOCKER_COMPOSE_FILE"
source "$DOCKER_CONFIG_ENV"
source "$DOCKER_IMAGE_ENV"

#### AI100
VG_NAME="prosuite-vg"
LV_NAME="prosuite-rd"

#### APT source
DOCKER_URL=https://download.docker.com/linux/ubuntu
DOCKER_KEYRING=/etc/apt/keyrings/docker.asc
DOCKER_APT=/etc/apt/sources.list.d/docker.list

NVIDIA_URL=https://nvidia.github.io
NVIDIA_KEYRING=/usr/share/keyrings/nvidia-container-toolkit-keyring.gpg
NVIDIA_APT=/etc/apt/sources.list.d/nvidia-container-toolkit.list
NVIDIA_VERSION=570

APT_PREPARE_LIST=("apt-transport-https" \
                  "ca-certificates" \
                  "curl" \
                  "gnupg" \
                  "lsb-release" \
                  "nvme-cli" \
                  "lvm2" \
                  "xfsprogs" \
                  "git-lfs" \
                  "jq" \
                  "logrotate" )

APT_PACKAGE_LIST=("docker-ce" \
                  "docker-compose-plugin" \
                  "nvidia-driver-$NVIDIA_VERSION-open" \
                  "libnvidia-container1:amd64=1.17.9-1" \
                  "libnvidia-container-tools=1.17.9-1" \
                  "nvidia-container-toolkit-base=1.17.9-1" \
                  "nvidia-container-toolkit=1.17.9-1" \
                  "nvidia-container-runtime" \
                  "nvtop" )


#### FW ####
FIRMWARE_INFO_FILE="$PROSUITE_INSTALL_DIR/pkg/fw/firmware_info.json"
if [[ ! -f "$FIRMWARE_INFO_FILE" ]]; then
    echo -e "$(date +%F\ %T) : File missing $FIRMWARE_INFO_FILE."
    exit 1
fi

#### Folder list
#### PHISONAI_DATA_SRC_PATH = /opt/phisonai/data in phison-config.env
FOLDER_LIST=("$PROSUITE_DATA_DIR"
             "$PROSUITE_DATA_DIR/chroma"
             "$PROSUITE_DATA_DIR/postgresql/data"
             "$PROSUITE_DATA_DIR/mongodb"
             "$PROSUITE_DATA_DIR/prometheus"
             "$PROSUITE_DATA_DIR/grafana"
             "$PROSUITE_DATA_DIR/rabbitmq"
             "$PHISONAI_DATA_SRC_PATH"
             "$PHISONAI_DATA_SRC_PATH/awq-log"
             "$PHISONAI_DATA_SRC_PATH/generate-dataset"
             "$PHISONAI_DATA_SRC_PATH/dataset"
             "$PHISONAI_DATA_SRC_PATH/service-model"
             "$PHISONAI_DATA_SRC_PATH/service-model/km-api"
             "$PHISONAI_DATA_SRC_PATH/benchmark"
             "$DEFAULT_MODEL_DST"
             "$PHISONAI_CACHE_PATH" )


#### Check whether the executing user is Root user
function check_root() {
    if [ "$(id -u)" != "0" ]; then
        msg1="Please execute this script with root privileges."
        msg2="Pro Suite not installed, Exited."
        echo -e "$(date +%F\ %T) : [error] [ENV-SWE-PRCE01] $msg1"
        echo -e "$(date +%F\ %T) : [error] [ENV-SWE-PRCE01] $msg2"
        exit 1
    else
        echo -e "$(date +%F\ %T) : [info] Confirm that script is started by the root user."
    fi
}


#### Check startup on boot
function check_start_onboot() {
    LOG "info" "Add cmd to crontab, startup on boot."
    str="docker compose --env-file $DOCKER_CONFIG_ENV --env-file $DOCKER_IMAGE_ENV --file $DOCKER_COMPOSE_FILE up -d"

    crontab -u root -l | grep "@reboot docker compose" | wc -l > /dev/null 2>&1
    if [[ $? -eq 0 ]]; then
        crontab -u root -l | grep -v "\@reboot docker compose" | crontab -u root - > /dev/null 2>&1
    fi
    
    crontab -u root -l | { cat; echo "@reboot $str >> $LOG_FILE ; crontab -u root -l | grep -v "\@reboot docker compose" | crontab -u root -"; } | crontab - > /dev/null 2>&1
}


#### Print log and to log file
function LOG() {
    logLevel=$1
    logMsg=$2
    echo -e "$(date +%F\ %T) : [$logLevel] $logMsg" >> "$LOG_FILE" 2>&1
    echo -e "$(date +%F\ %T) : [$logLevel] $logMsg"
}


#### Check if the folder and file exists
function check_log_exist() {
    if [ ! -f "$LOG_FILE" ]; then
        /usr/bin/touch "$LOG_FILE"
    fi
}


#### If yad does not exist, the user is asked to choose whether to install it.
function check_yad() {
    if dpkg -s yad &> /dev/null; then
        :
    else
        LOG "info" "Installation requires the use of the package yad."
        read -p "Do you want to installed yad (y/n) ? " installYad

        if [[ "$installYad" == "y" || "$installYad" == "Y" ]] ;then
            apt_update
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            apt install -y yad
            if dpkg -s yad &> /dev/null; then
                :
            else
                LOG "error" "[ENV-SWE-PRCE04] Package yad is not install corrected."
            fi
        else
            LOG "error" "[ENV-SWE-PRCE10] Required apt install yad pacakge to start Pro Suite installation."
            return 1
        fi
    fi
}


#### Check bios mode, must UEFI
function check_bios_mode() {
    if [[ -d "/sys/firmware/efi" ]]; then
        LOG "info" "UEFI Boot Mode - Legacy mode is disabled."
    else
        LOG "info" "UEFI Boot Mode - Legacy mode is enabled."

        msg1="[ENV-HWE-PRCE01] Bios mode 'lagecy' is enabled on this machine."
        msg2="To disable lagecy in the BIOS:"
        msg3="1. Start or restart your computer."
        msg4="2. Press the key to enter the BIOS setup during startup (F2, F10, Del, Esc, etc.)."
        msg5="3. Navigate to the Boot Settings tab."
        msg6="4. Find Boot mode or Boot options and set it to Disabled or Off."
        msg7="5. Save changes and exit the BIOS setup."
        msg8="6. Your computer will restart with Secure Boot disabled."

        if [[ "$MODE" == "gui" ]]; then
            yad_paned "warning" "Warning" "$msg1\n\n$msg2\n$msg3\n$msg4\n$msg5\n$msg6\n$msg7\n$msg8\n"
        fi
        
        LOG "error" "$msg1"
        LOG "info" "$msg2"
        LOG "info" "$msg3"
        LOG "info" "$msg4"
        LOG "info" "$msg5"
        LOG "info" "$msg6"
        LOG "info" "$msg7"
        LOG "info" "$msg8"
        exit 1
    fi
}


#### Check bios secure boot disable
function check_secure_boot() {
    if dpkg -s mokutil &> /dev/null; then
        :
    else
        apt install -y mokutil &> /dev/null
    fi

    secure_boot_enabled=$(mokutil --sb-state | grep -i "SecureBoot enabled")
    if [[ "$secure_boot_enabled" ]]; then
        msg1="Secure boot is enabled on this machine."
        msg2="To disable Secure Boot in the BIOS:"
        msg3="1. Start or restart your computer."
        msg4="2. Press the key to enter the BIOS setup during startup (F2, F10, Del, Esc, etc.)."
        msg5="3. Navigate to the Security tab."
        msg6="4. Find Secure Boot or UEFI Secure Boot and set it to Disabled or Off."
        msg7="5. Save changes and exit the BIOS setup."
        msg8="6. Your computer will restart with Secure Boot disabled."

        if [[ "$MODE" == "gui" ]]; then
            yad_paned "warning" "Warning" "$msg1\n\n$msg2\n$msg3\n$msg4\n$msg5\n$msg6\n$msg7\n$msg8\n"
        fi

        LOG "error" "[ENV-HWE-PRCE02] $msg1"
        LOG "error" "[ENV-HWE-PRCE02] $msg2"
        LOG "error" "[ENV-HWE-PRCE02] $msg3"
        LOG "error" "[ENV-HWE-PRCE02] $msg4"
        LOG "error" "[ENV-HWE-PRCE02] $msg5"
        LOG "error" "[ENV-HWE-PRCE02] $msg6"
        LOG "error" "[ENV-HWE-PRCE02] $msg7"
        LOG "error" "[ENV-HWE-PRCE02] $msg8"
        exit 1
    fi
}


#### YAD Notify
function yad_paned() {
    yadLevel=$1
    yadTitle=$2
    yadText=$3

    yad --"$yadLevel" \
        --width=500 --height=120 --center \
        --title="$yadTitle" \
        --text="$yadText" \
        --button="OK:0"
}


#### After startup
function install_finished() {
    if [[ "$MODE" == "gui" ]]; then
        msg="Phison ai100 disk is mount at $PHISONAI_CACHE_PATH."
        msg+="\n\nPro Suite has been launched, you can access Pro Suite service via below URL:"
        msg+="\nhttp://localhost:$FRONTEND_LISTEN_PORT_ON_HOST"
        msg+="\nhttp://127.0.0.1:$FRONTEND_LISTEN_PORT_ON_HOST"

        IFS=' ' read -r -a ip_array <<< "$(hostname -I)"
        for ip_addr in "${ip_array[@]}"; do
            msg+="\nhttp://$ip_addr:$FRONTEND_LISTEN_PORT_ON_HOST"
        done

        msg+="\n\nLogin account: admin@aidaptiv.com"
        msg+="\nLogin password: Admin8299"
        msg+="\nSuggest to reset the default password on your first log-in."
        yad_paned "info" "Installation finished" "$msg"
    fi

    LOG "info" "http://localhost:$FRONTEND_LISTEN_PORT_ON_HOST"
    LOG "info" "http://127.0.0.1:$FRONTEND_LISTEN_PORT_ON_HOST"

    IFS=' ' read -r -a ip_array <<< "$(hostname -I)"
    for ip_addr in "${ip_array[@]}"; do
        LOG "info" "http://$ip_addr:$FRONTEND_LISTEN_PORT_ON_HOST"
    done
}


function check_nv_install_time() {
    LASTBOOT_DATE=$(who -b | awk '{print $3, $4}' | xargs -I {} date -d "{}" +%s)
    LOG "info" "System last boot time: $(date +'%Y-%m-%d %H:%M:%S' -d @$LASTBOOT_DATE)"

    APT_CHECK_LIST=("nvidia-driver-$NVIDIA_VERSION-open" "nvidia-container-runtime")
    REBOOT_FLAG=false

    for pkg in "${APT_CHECK_LIST[@]}"; do
        PKG_INSTALL_DATE=$(stat -c %Y /var/lib/dpkg/info/$pkg.list 2>/dev/null | sort -n | tail -n 1)
        if [[ -z "$PKG_INSTALL_DATE" ]]; then
            LOG "warning" "ENV-IST-A00004 The installation time of $pkg cannot be found."
            return 1
        else
            LOG "info" "$pkg installed time $(date +'%Y-%m-%d %H:%M:%S' -d @$PKG_INSTALL_DATE)"
        fi

        if [[ "$PKG_INSTALL_DATE" -gt "$LASTBOOT_DATE" ]]; then
            REBOOT_FLAG=true
        fi
    done
}


#### Check if need reboot
function check_reboot() {
    check_nv_install_time
    if [[ $? -ne 0 ]]; then
        return 1
    fi

    $DOCKER_CLI create
    if [[ "$?" -eq 0 ]]; then
        LOG "info" ""
        LOG "info" "The service will automatically start after the next reboot."
    else
        if [[ "$MODE" == "gui" ]]; then
            yad_paned "error" "Failed" "Failed create Pro Suite Service\n$($DOCKER_CLI create)."
        fi
        LOG "error" "Failed create Pro Suite Service $($DOCKER_CLI create)."
    fi

    if [[ "$REBOOT_FLAG" == "true" ]]; then
        ## Add autostart on boot
        check_start_onboot

        LOG "info" "The system needs to be rebooted to enable the nvidia driver"
        msg="Reboot the machine to apply nvidia driver ?"
        if [[ "$MODE" == "gui" ]]; then
            yad --question \
                --width=400 --height=150 --center \
                --title="Reboot machine" \
                --text="$msg"

            result=$?
            if [[ "$result" -eq 0 ]]; then
                LOG "info" "Reboot system"
                reboot now
            fi
        elif [[ "$MODE" == "cli" ]]; then
            result=$(ask_question "$msg")
            if [[ "$result" -eq "0" ]]; then
                LOG "info" "Reboot system"
                reboot now
            fi
        fi
    fi
}


#### Folder ####
function init_directory() {
    LOG "info" "Check folder and permission."

    for folder in "${FOLDER_LIST[@]}"; do
        if [[ ! -d "$folder" ]]; then
            LOG "info" "Path is not exist, create path $folder."
            mkdir -p "$folder"
        fi

        # Permission adjustment, PostgreSQL folder must user userid=1001
        if [[ $folder == *postgresql* ]]; then
            chown 1001:1001 -R "$folder"
        else
            chown "$USER":"$USER" "$folder"
        fi
        chmod 755 "$folder"

        # Copy Benchmark.py to specific folder
        if [[ $folder == "$PHISONAI_DATA_SRC_PATH/benchmark" ]]; then
            cp "$PROSUITE_INSTALL_DIR"/config/benchmark/benchmark.py "$PHISONAI_DATA_SRC_PATH"/benchmark/
        fi
    done
}


function remove_prosuite_service_data() {
    if [[ -d $PROSUITE_DATA_DIR ]]; then
        LOG "info" "Remove folder $PROSUITE_DATA_DIR."
        rm -rf "$PROSUITE_DATA_DIR"
    fi

    if [[ -d "/opt/phisonai/data" ]]; then
        LOG "info" "Remove folder /opt/phisonai/data/*."
        rm -rf /opt/phisonai/data/*
    fi
}


#### APT relate ####
function apt_update() {
    LOG "info" "Executing apt update."
    apt update &> "$LOG_FILE" 2>&1
    if [[ $? -ne 0 ]]; then
        LOG "error" "[ENV-SWE-PRCE03] An error occurred during apt update execution, please check /etc/apt/sources or the status of the computer network connection"
        return 1
    fi

    apt-get install -y --fix-missing > $LOG_FILE
    if [[ $? -ne 0 ]]; then
        LOG "error" "[ENV-SWE-PRCE04] Failed apt fix-missing."
        return 1
    fi
}


function install_packages() {
    for package in "${@}"; do
        # Extract package name and version (if specified)
        package_name="${package%=*}"
        
        # Check if package needs to be installed/downgraded/upgraded
        if [[ "$package" == *"="* ]]; then
            # Version specified format: package=version
            expected_version="${package#*=}"
            
            # Get current installed version
            current_version=$(dpkg -l 2>/dev/null | grep "^ii" | awk -v pkg="$package_name" '{if ($2==pkg) print $3}')
            
            if [[ -z "$current_version" ]]; then
                # Package not installed
                LOG "info" "Install apt package $package."
                apt-get install -y "$package" > $LOG_FILE
            elif [[ "$current_version" != "$expected_version" ]]; then
                # Version mismatch - force install correct version
                LOG "info" "Package $package_name version mismatch. Current: $current_version, Expected: $expected_version. Reinstalling..."
                apt-get install -y "$package" > $LOG_FILE
            else
                # Version matches
                LOG "info" "Package $package_name version $current_version already installed correctly."
                continue
            fi
            
            result=$?
            if [[ "$result" -ne 0 ]]; then
                LOG "error" "ENV-IST-A00002 Failed apt install $package."
                return 1
            fi
        else
            # No version specified - use original logic
            if dpkg -s "$package_name" &> /dev/null; then
                LOG "info" "Package $package_name already installed."
            else
                LOG "info" "Install apt package $package."
                apt-get install -y "$package" > $LOG_FILE

                result=$?
                if [[ "$result" -ne 0 ]]; then
                    LOG "error" "ENV-IST-A00002 Failed apt install $package."
                    return 1
                fi
            fi
        fi
    done
}


function apt_purge(){
    local item=$1
    if dpkg --status $item >/dev/null 2>&1; then
        apt-get purge -y $item >> "$LOG_FILE" 2>&1
    fi
}


function disable_auto_upgrade() {
    apt_purge "unattended-upgrades"
    apt_purge "update-manager-core"

    if [ -f /etc/update-manager/release-upgrades ]; then
        sed -i 's/Prompt=.*/Prompt=never/' /etc/update-manager/release-upgrades
    fi

    if [ -f /etc/apt/apt.conf.d/10periodic ]; then
        sed -i 's/APT::Periodic::Update-Package-Lists "1";/APT::Periodic::Update-Package-Lists "0";/' /etc/apt/apt.conf.d/10periodic
    fi
}


function apt_before_install() {
    # 安裝 APT_PREPARE_LIST
    for package in "${APT_PREPARE_LIST[@]}"; do
        install_packages "$package"
        if [[ $? -ne 0 ]]; then
            return 1
        fi
    done
}


function update_ssd_firmware() {
    json_data=$(cat "$FIRMWARE_INFO_FILE")
    device_count=$(echo "$json_data" | jq '.devices | length')

    for ((i=0; i<$device_count; i++)); do
        # 可能有多種硬碟, e.g ai100, ai200
        PHISON_FW_ID=$(echo "$json_data" | jq -r ".devices[$i].fw_id")
        TARGET_FW_VERSION=$(echo "$json_data" | jq -r ".devices[$i].version")
        FW_BIN_PATH=$PROSUITE_INSTALL_DIR/config/fw/$(echo "$json_data" | jq -r ".devices[$i].bin_path")

        # 檢查是否有匹配的SSD
        if [[ -z "$(nvme list | grep -i "$PHISON_FW_ID")" ]]; then
            continue
        fi

        if [[ ! -f "$FW_BIN_PATH" ]]; then
           LOG "error" "ENV-PRC-E00002" "File missing $FW_BIN_PATH."
           exit 1
        fi

        # 更新韌體
        while read -r line; do
            DEVICE=$(echo "$line" | awk '{print $1}')
            CURRENT_FW_VERSION=$(nvme id-ctrl "$DEVICE" | grep -i "$PHISON_FW_ID" | awk '{print $3}')


            ## 61.4 -> 61T4 -> 61.5
            NEED_BRIDGE=$(echo "$json_data" | jq -r ".need_bridge")
            BRIDGE_VERSION=$(echo "$json_data" | jq -r ".bridge_version")
            BRIDGE_BIN_PATH=$PROSUITE_INSTALL_DIR/config/fw/$(echo "$json_data" | jq -r ".bridge_bin_path")

            if [[ "$CURRENT_FW_VERSION" == "$NEED_BRIDGE" ]]; then
                if [[ ! -f "$BRIDGE_BIN_PATH" ]];then
                    LOG "error" "ENV-PRC-E00002" "File missing $BRIDGE_BIN_PATH."
                    exit 1
                fi
                LOG "info" "Update $DEVICE firmware version from $CURRENT_FW_VERSION to $BRIDGE_VERSION due bridge."
                if nvme fw-download "$DEVICE" --fw="$BRIDGE_BIN_PATH" > /dev/null ;then
                    if nvme fw-commit "$DEVICE" --slot=1 --action=3 > /dev/null; then
                        LOG "info" "Firmware update completed $DEVICE."
                    else
                        LOG "error" "ENV-DSK-E00012" "Fail fw-commit $BRIDGE_BIN_PATH to $DEVICE." 
                        exit 1
                    fi
                else
                    LOG "error" "ENV-DSK-E00012" "Fail fw-download $BRIDGE_BIN_PATH to $DEVICE." 
                    exit 1
                fi
            fi

	    CURRENT_FW_VERSION=$(nvme id-ctrl "$DEVICE" | grep -i "$PHISON_FW_ID" | awk '{print $3}')
            ## other -> 61.5
            if [[ "$CURRENT_FW_VERSION" != "$TARGET_FW_VERSION" ]]; then
                LOG "info" "Update $DEVICE firmware version from $CURRENT_FW_VERSION to $TARGET_FW_VERSION."
                if nvme fw-download "$DEVICE" --fw="$FW_BIN_PATH" > /dev/null; then
                    if nvme fw-commit "$DEVICE" --slot=1 --action=3 > /dev/null; then
                        LOG "info" "Firmware update completed $DEVICE."
                    else
                        LOG "error" "ENV-DSK-E00012" "Fail fw-commit $FW_BIN_PATH to $DEVICE."
                        exit 1
                    fi
                else
                    LOG "error" "ENV-DSK-E00012" "Fail fw-download $FW_BIN_PATH to $DEVICE."
                    exit 1
                fi
            fi
        done < <(nvme list | grep -i "$PHISON_FW_ID")
    done
}


# 檢查SSD韌體版本
function check_if_match_expected_fw_version() {
    json_data=$(cat "$FIRMWARE_INFO_FILE")
    device_count=$(echo "$json_data" | jq '.devices | length')
    phison_device_count=0

    for ((i=0; i<$device_count; i++)); do
        PHISON_FW_ID=$(echo "$json_data" | jq -r ".devices[$i].fw_id")
        TARGET_FW_VERSION=$(echo "$json_data" | jq -r ".devices[$i].version")

        # 檢查是否有匹配的SSD
        if [[ -z "$(nvme list | grep -i "$PHISON_FW_ID")" ]]; then
            continue
        fi

        while read -r line; do
            DEVICE=$(echo "$line" | awk '{print $1}')
            CURRENT_FW_VERSION=$(nvme id-ctrl "$DEVICE" | grep -i "$PHISON_FW_ID" | awk '{print $3}')
            LOG "info" "Check Phison disk device $DEVICE."
            LOG "info" "Target firmware version is $TARGET_FW_VERSION."
            LOG "info" "Current firmware version is $CURRENT_FW_VERSION."
            if [[ "$CURRENT_FW_VERSION" != "$TARGET_FW_VERSION" ]]; then
                return 1
            fi
            phison_device_count=$((phison_device_count+1))
        done < <(nvme list | grep -i "$PHISON_FW_ID")

    done

    if [[ $phison_device_count -eq 0 ]]; then
        LOG "error" "ENV-DSK-W00001" "No matching SSDs found."
        exit 1
    fi
}


# 韌體更新初始化
function update_firmware_startup() {
    packages=("jq" "nvme-cli")
    for package in "${packages[@]}"; do
        if ! dpkg -s $package &> /dev/null; then
            if ! apt_update; then
                exit 1
            fi

            if ! install_packages $package; then
                exit 1
            fi
        fi
    done

    if ! check_if_match_expected_fw_version; then
        if [[ "$UPGRADE_FW" == "FALSE" || "$UPGRADE_FW" == "False" || "$UPGRADE_FW" == "false" ]];then
            return 0
        elif [[ "$UPGRADE_FW" == "TRUE" || "$UPGRADE_FW" == "True" || "$UPGRADE_FW" == "true" ]];then
            result=0
        else
            msg="Upgrade SSD firmware?"
            result=$(ask_question "$msg")
        fi

        if [[ "$result" -eq 0 ]]; then
            LOG "info" "Confirm upgrade SSD firmware"
            update_ssd_firmware
        else
            msg1="It is recommended to update the SSD firmware version."
            msg2="aiDAPTIV cannot run under mismatched SSD firmware versions."
            LOG "info" "ENV-DSK-W00003" "$msg1"
            LOG "error" "ENV-DSK-W00003" "$msg2"
            exit 1
        fi
    fi
}


function apt_after_install() {
    # 安裝 APT_PACKAGE_LIST
    for package in "${APT_PACKAGE_LIST[@]}"; do
        LOG "info" "Check apt package: $package"
        
        # 提取包名（去除版本指定）
        package_name="${package%=*}"
        
        if dpkg -s "$package_name" &> /dev/null; then
             LOG "info" "Package $package_name already installed."
             continue
        else
            # Check whether other nv driver versions are installed
            if [[ $package =~ "nvidia-driver" ]]; then
                install_packages "$package"
                if [[ $? -ne 0 ]]; then
                    return 1
                else
                    apt-mark hold "$package_name" &> /dev/null
                    LOG "info" "apt-mark hold $package_name"
                fi

            else
                install_packages "$package"
                if [[ $? -ne 0 ]]; then
                    return 1
                fi
                
                # Lock version for NVIDIA Container Toolkit related packages
                if [[ $package =~ "libnvidia-container" ]] || [[ $package =~ "nvidia-container-toolkit" ]]; then
                    apt-mark hold "$package_name" &> /dev/null
                    LOG "info" "apt-mark hold $package_name"
                fi
                
                # Also lock nvidia-container-runtime
                if [[ $package == "nvidia-container-runtime" ]]; then
                    apt-mark hold "nvidia-container-runtime" &> /dev/null
                    LOG "info" "apt-mark hold nvidia-container-runtime"
                fi
            fi
        fi
    done
}


function add_docker_apt_key() {
    ## Docker GPG key
    LOG "info" "Get docker gpg key from $DOCKER_URL/gpg"

    if [[ ! -f $DOCKER_KEYRING ]]; then
        curl -kfsSL $DOCKER_URL/gpg -o $DOCKER_KEYRING > /dev/null 2>&1
        result=$?
        if [[ "$result" -eq 0 ]]; then
            chmod a+r "$DOCKER_KEYRING"
        else
            rm $DOCKER_KEYRING > /dev/null 2>&1
            msg="ENV-IST-A00003 Failed add docker apt gpg key."
            if [[ "$MODE" == "gui" ]]; then
                yad_paned "warning" "Get Docker apt gpg key" "$msg"
            fi
            LOG "warning" "$msg"
            exit 1
        fi
    fi

    ## Docker apt repository
    LOG "info" "Get docker apt repository $DOCKER_APT"

    if [[ -f $DOCKER_APT ]]; then
        LOG "info" "Recreate docker apt source file $DOCKER_APT."
        rm "$DOCKER_APT"
    fi

    echo \
    "deb [arch=$(dpkg --print-architecture) signed-by=$DOCKER_KEYRING] $DOCKER_URL \
    $(. /etc/os-release && echo "$VERSION_CODENAME") stable" | \
    tee $DOCKER_APT > /dev/null

    result=$?
    if [[ "$result" -ne 0 ]]; then
        rm $DOCKER_APT > /dev/null 2>&1
        msg="ENV-IST-A00003 Failed add docker apt repository."
        if [[ "$MODE" == "gui" ]]; then
            yad_paned "warning" "Docker apt failed" "$msg"
        fi
        LOG "warning" "$msg"
    fi
}


function add_nvidia_apt_key() {
    ## NVIDIA GPG key
    LOG "info" "Get nvidia gpg key from $NVIDIA_URL/libnvidia-container/gpgkey"
    if [[ ! -f "$NVIDIA_KEYRING" ]]; then
        curl -kfsSL "$NVIDIA_URL"/libnvidia-container/gpgkey | gpg --dearmor -o "$NVIDIA_KEYRING" > /dev/null 2>&1
        result=$?

        if [[ "$result" -eq 0 ]]; then
            chmod a+r "$NVIDIA_KEYRING"
        else
            rm "$NVIDIA_KEYRING" > /dev/null 2>&1
            msg="ENV-IST-A00003 Failed add nvidia apt gpg key."
            if [[ "$MODE" == "gui" ]]; then
                yad_paned "warning" "Get NVIDIA apt gpg key" "$msg"
            fi
            LOG "warning" "$msg"
            exit 1
        fi
    fi

    ## NVIDIA apt repository
    LOG "info" "Get nvidia apt repository $NVIDIA_APT"
    if [[ ! -f $NVIDIA_APT ]]; then
        curl -ksL $NVIDIA_URL/libnvidia-container/stable/deb/nvidia-container-toolkit.list | \
        sed "s#deb https://#deb [signed-by=$NVIDIA_KEYRING] https://#g" | \
        tee $NVIDIA_APT > /dev/null
        result=$?

        if [[ "$result" -ne 0 ]]; then
            rm $NVIDIA_APT > /dev/null 2>&1
            msg="ENV-IST-A00003 Failed add nvidia apt repository."
            if [[ "$MODE" == "gui" ]]; then
                yad_paned "warning" "NVIDIA apt failed" "$msg"
            fi
            LOG "warning" "$msg"
        fi
    fi
}


#### Remove apt source
function remove_apt_sourcelist() {
    files=("$NVIDIA_KEYRING" "$NVIDIA_APT" "$DOCKER_KEYRING" "$DOCKER_APT")
    for file in "${files[@]}"; do
        if [[ -f "$file" ]]; then
            LOG "info" "Remove apt source list $file."
            rm "$file"
        fi
    done
}


#### Docker relate ####
function config_docker() {
    LOG "info" "Config docker /etc/docker/daemon.json"
    if [[ -d /etc/docker ]]; then
        if [[ ! -f /etc/docker/daemon.json ]]; then
            cp "$PROSUITE_INSTALL_DIR"/config/docker/daemon.json /etc/docker/daemon.json
        fi
    fi
}


#### Systemctl start docker
function start_docker_service() {
    LOG "info" "Starting docker service"
    systemctl start docker >> /dev/null
    result=$?
    if [[ "$result" -ne 0 ]]; then
        LOG "error" "ENV-IST-C00007 Failed start docker $(systemctl start docker)"
        return 1
    fi
}


#### Create phison-network
function create_docker_network() {
    if ! docker network inspect "$DOCKER_NETWORK" > /dev/null 2>&1; then
        LOG "info" "Create docker network $DOCKER_NETWORK."
        
        if docker network create "$DOCKER_NETWORK"; then
            subnet=$(docker network inspect $DOCKER_NETWORK | jq -r '.[0].IPAM.Config[0].Subnet')
            LOG "info" "Success create $DOCKER_NETWORK with subnet $subnet."
        else
            LOG "info" "ENV-IST-C00005 Failed create $DOCKER_NETWORK with subnet $subnet."
            exit 1
        fi
    fi
}


#### Remove phison-network
function remove_docker_network() {
    LOG "info" "Remove docker network $DOCKER_NETWORK."

    docker network inspect "$DOCKER_NETWORK" > /dev/null 2>&1
    result=$?
    if [[ "$result" -eq 0 ]]; then
        docker network rm "$DOCKER_NETWORK" >> /dev/null

        result=$?
        if [[ "$result" -eq 0 ]]; then
            LOG "info" "Success remove docker network $DOCKER_NETWORK."
        else
            LOG "warning" "ENV-IST-C00006 Failed to remove docker network $DOCKER_NETWORK."
        fi
    fi
}


#### Normal install
#### Pull docker image from docherhub
function get_docker_image() {
    docker_login
    result=$?
    if [[ "$result" -ne 0 ]]; then
        exit 1
    fi

    current_image=1
    total_images=$(grep -c "IMAGE" "$DOCKER_IMAGE_ENV")
    LOG "info" "Total image is $total_images."

    while IFS= read -r line; do
        if [[ $line != *"IMAGE"* ]]; then
            continue
        fi

        image=$(echo "$line" | cut -d '=' -f2)
        LOG "info" "Pulling the $current_image/$total_images docker image $image."
        docker pull --quiet "$image" 1>&2 > /dev/null

        if [[ $? -ne 0 ]]; then
            LOG "error" "ENV-IST-C00003 Fail pull image $image."
            return 1
        fi

        current_image=$((current_image + 1))
    done < "$DOCKER_IMAGE_ENV"

    docker_logout
}

#### Normal install
#### Docker login
function docker_login() {
    attempt=1
    while [[ $attempt -le 3 ]]; do
        LOG "info" "Try docker login, attempt $attempt."

        echo "$PROSUITE_IMAGE_TOKEN" | docker login -u "$PROSUITE_IMAGE_USER" --password-stdin > /dev/null 2>&1

        result=$?
        if [[ "$result" -eq 0 ]]; then
            LOG "info" "Successful login docker."
            return 0
        else
            LOG "error" "ENV-IST-C00001 Failed login docker,  Try $attempt/3 times."
        fi
        ((attempt++))
    done

    if [[ "$MODE" == "gui" ]]; then
        yad_paned "error" "Failed"  "ENV-IST-C00001 Failed docker login."
    fi
    LOG "error" "ENV-IST-C00001 Failed docker login."
    return 1
}

#### Normal install
#### Docker logout
function docker_logout() {
    docker logout "$PROSUITE_IMAGE_USER" > /dev/null 2>&1
    LOG "error" "ENV-IST-C00002 Failed docker login."
}


#### Update ai100 mount path
function update_env_cache_path() {
    ENV_PATH=$(echo "$PHISONAI_CACHE_PATH" | sed 's/\//\\\//g')
    sed -i "s/^PHISONAI_CACHE_PATH.*/PHISONAI_CACHE_PATH=$ENV_PATH/g" "$DOCKER_CONFIG_ENV"
    LOG "info" "Update aiDAPTIV cache path $PHISONAI_CACHE_PATH to $DOCKER_CONFIG_ENV."
}


#### Update timezone
function update_timezone() {
    tz=$(cat /etc/timezone)
    tzEnv=$(sed -e 's|/|\\/|g' /etc/timezone)
    sed -i "s/^TZ=.*/TZ=$tzEnv/g" "$DOCKER_CONFIG_ENV"
    LOG "info" "Update timezone $tz to $DOCKER_CONFIG_ENV."
}


#### Update initramfs
function update_initramfs() {
    LOG "info" "Update the initramfs of the current kernel $(uname -r)."
    update-initramfs -u > /dev/null 2>&1
}


function update_grub() {
    LOG "info" "Update the grub of the current kernel $(uname -r)."
    update-grub > /dev/null 2>&1
}


#### Use 3% of the host's memory as container shm
function update_memsize() {
    totalMemory=$(grep MemTotal /proc/meminfo | awk '{print $2}')
    threePercentMemory=$(echo "scale=2; $totalMemory * 0.03 * 1024" | bc)
    shmSizeByte=$(printf "%.0f" "$threePercentMemory")
    LOG "info" "Host total memory: $totalMemory KB."
    LOG "info" "Use $shmSizeByte bytes (3%) as container share memory."

    sed -i "s/SWAPFILE_SIZE_BYTE=.*/SWAPFILE_SIZE_BYTE=$shmSizeByte/g" "$DOCKER_CONFIG_ENV"
    result=$?
    if [[ "$result" -eq 0 ]]; then
        LOG "info" "Update share memory info to $DOCKER_CONFIG_ENV."
    else
        LOG "warning" "ENV-IST-P00004 Failed update shm info to $DOCKER_CONFIG_ENV."
    fi
}


function check_backend() {
    for i in {1..18}; do
        response=$(curl -s -o /dev/null -w "%{http_code}" -X 'GET' 'http://127.0.0.1:8799/api/Test/Backend' -H 'accept: text/plain')
        if [ $response == "200" ]; then
            LOG "info" "Pro Suite web-backend is workable."
            install_finished
            return 0
        else
            LOG "info" "Curl http://127.0.0.1:8799/api/Test/Backend get response code: $response."
            LOG "info" "Waiting web-backend started. Try $i/18..."
            sleep 10
        fi
    done
    return 1
}


#### Docker compose ####
function dc_startup() {
    check_nv_install_time

    if [[ "$REBOOT_FLAG" == "true" ]]; then
        LOG "info" "Need to restart the system to continue."
        return 1
    fi

    $DOCKER_CLI up -d
    result=$?

    if [[ "$result" -eq 0 ]]; then
        if check_backend; then
            return 0
        else
            LOG "error" "ENV-IST-P00005 Failed start Pro Suite Web backend Service."
            return 1
        fi
    else
        if [[ "$MODE" == "gui" ]]; then
            yad_paned "error" "Failed" "ENV-IST-C00008 Failed start Pro Suite Service\n$($DOCKER_CLI up -d)."
        fi
        LOG "error" "ENV-IST-C00008 Failed start Pro Suite Service $($DOCKER_CLI up -d)."
    fi
}


function dc_stop() {
    remove_image=$1
    service_list=($($DOCKER_CLI ps --services))

    ## If service list not empty, means docker compose can identify pro suite
    if [ ${#service_list[@]} -ne 0 ]; then
        if [[ "$remove_image" == "rmi" ]];then
            LOG "info" "Stop Pro Suite services and remove inuse docker images."
            $($DOCKER_CLI down --remove-orphans --rmi all)
        else
            LOG "info" "Stop Pro Suite services by docker compose."
            $($DOCKER_CLI down --remove-orphans)
        fi

        result=$?
        if [[ "$result" -ne 0 ]]; then
            if [[ "$MODE" == "gui" ]]; then
                yad_paned "error" "Failed" "ENV-IST-C00008 Failed stop Pro Suite Service"
            fi
            LOG "error" "ENV-IST-C00008 Failed stop Pro Suite Service"
        fi
    fi

    ## Remove by Docker network
    docker network inspect "$DOCKER_NETWORK" > /dev/null 2>&1
    result=$?
    if [[ "$result" -eq 0 ]]; then
        LOG "info" "Stop Pro Suite services by docker network."
        container_names=($(docker network inspect $DOCKER_NETWORK | jq '.[0].Containers | to_entries | .[] | .value.Name' | tr -d '"'))
        for container in "${container_names[@]}"; do
            LOG "info" "Stop docker container ${container}."
            docker rm -f "${container}" > /dev/null 2>&1
        done
    fi

    ## Check by name
    LOG "info" "Check docker container by name."
    while IFS= read -r line; do
        CID=$(docker ps -a -q -f name=$line)
        if [[ ! -z "$CID" ]]; then
            LOG "info" "Remove container $line."
            docker rm -f $line > /dev/null 2>&1
        fi
    done < <(cat $DOCKER_COMPOSE_FILE | grep container_name | awk '{print $2}')
}


function dc_status() {
    DC_STATUS=$($DOCKER_CLI ps --format json)

    result=$?
    if [ "$result" -ne 0 ]; then
        if [ "$MODE" == "gui" ]; then
            yad_paned "error" "Failed" "ENV-IST-C00008 Failed get Pro Suite Service"
        fi
        LOG "error" "ENV-IST-C00008 Failed get Pro Suite Service"
        return 1
    fi

    container_ids=($(echo "$DC_STATUS" | jq -r '.ID'))
    names=($(echo "$DC_STATUS" | jq -r '.Name'))
    state=($(echo "$DC_STATUS" | jq -r '.State'))
    status=($(echo "$DC_STATUS" | jq -r '.Status' | awk '{print $1}'))
    ports=($(echo "$DC_STATUS" | jq -r '.Ports' | sed 's/ //g'))

    yad_info=()
    for ((i=0; i<${#container_ids[@]}; i++)); do
        yad_info+=("${container_ids[i]}" "${names[i]}" "${state[i]}" "${status[i]}" "${ports[i]}")
    done

    if [ "$MODE" == "gui" ]; then
        yad --list \
            --width=800 --height=400 --center \
            --no-click --no-selection \
            --title="Pro Suite service status" \
            --column="Container ID" --column="Name" --column="State" --column="Status" --column="Ports" "${yad_info[@]}" \
            --button="Previous:0"
    elif [ "$MODE" == "cli" ]; then
        printf "%15s %20s %10s %10s %10s %20s\n" "Container ID" "Name" "State" "Status" "Ports"
        for i in "${!container_ids[@]}"; do
            printf "%15s %20s %10s %10s %10s %20s\n" "${container_ids[i]}" "${names[i]}" "${state[i]}" "${status[i]}" "${ports[i]}"
        done
    fi
}


function dc_restart_specific() {
    serviceList=($($DOCKER_CLI ps --services))

    dialogarray=()
    for index in "${!serviceList[@]}"; do
        dialogarray+=("$index" "${serviceList[$index]}")
    done

    if [ "$MODE" == "gui" ]; then
        selected_service=$(yad --list \
                            --print-column=2 \
                            --separator="," \
                            --width=300 --height=400 --center \
                            --title="Select service" \
                            --text="Select the service to restart." \
                            --column="Num" --column="Service" "${dialogarray[@]}" \
                            --button="Previous:1" \
                            --button="Restart:0" )

    elif [ "$MODE" == "cli" ]; then
        for i in "${!serviceList[@]}"; do
            echo "$i ${serviceList[$i]}"
        done
        read -p "Please choose the service: " selected
        selected_service=${serviceList[${selected}]}
    fi

    result=$?
    if [[ "$result" -eq 0 ]]; then
        if [[ "$selected_service" != "" ]]; then
            selected_service=$(echo $selected_service | sed 's/\(,\s*\)+/,/g')
            IFS=',' read -r -a restart_service <<< "$selected_service"
            LOG "info" "Restart $restart_service"

            $DOCKER_CLI stop $restart_service > /dev/null 2>&1
            sleep 2
            $DOCKER_CLI up -d $restart_service > /dev/null 2>&1

            if [ "$MODE" == "gui" ]; then
                yad_paned "info" "Finished" "Finish restart service $restart_service."
            fi
            LOG "info" "Finish restart service $restart_service."
        else
            if [ "$MODE" == "gui" ]; then
                yad_paned "info" "Empty" "Selected empty"
            fi
            LOG "info" "Selected empty in restart option."
        fi
    fi
}


#### LVM RAID ####
#### Get available disk list from lsblk
function get_available_disk() {
    local -n returnList=$1
    inusedBlkDevices=()

    ## Check inused block devices
    blkList=$(lsblk -o NAME,MOUNTPOINT,TYPE -dn | awk '{print $1}' | grep -v "sr\|loop" | grep nvme)
    for blkDevice in $blkList; do
        mountPoint=$(lsblk -no MOUNTPOINT /dev/$blkDevice)
        if [[ -n "$mountPoint" ]]; then
            inusedBlkDevices+=($blkDevice)
        fi
    done

    ## Get unused block devices
    unusedBlkDevices=($(comm -23 <(echo "$blkList" | tr ' ' '\n' | sort) <(echo "${inusedBlkDevices[@]}" | tr ' ' '\n' | sort)))
    for blkIndex in "${!unusedBlkDevices[@]}"; do
        blkDeviceModel=$(lsblk -P -o MODEL /dev/${unusedBlkDevices[$blkIndex]} | grep -v "\"\"" | awk -F\" '{print $2}')
        blkDeviceSize=$(lsblk -P -o SIZE /dev/${unusedBlkDevices[$blkIndex]} | grep -v "\"\"" | awk -F\" '{print $2}' | head -n 1)

        ## Filter only nvme
        nvmeFr=$(nvme id-ctrl /dev/${unusedBlkDevices[$blkIndex]} -o json | jq -r '.fr')
        for id in $(jq -r '.devices[].fw_id' $FIRMWARE_INFO_FILE); do
            if [[ "$nvmeFr" == "$id"* ]]; then
                availableDiskList+=("" "$blkIndex" "/dev/${unusedBlkDevices[$blkIndex]}" "$blkDeviceModel" "$blkDeviceSize")
                break
            fi
        done
    done

    returnList="$availableDiskList"
}


function check_lvm() {
    ## If LVM already exist
    if lvdisplay /dev/$VG_NAME/$LV_NAME &> /dev/null; then
        LOG "info" "Logical volume /dev/$VG_NAME/$LV_NAME already exist."

        lvDmPath=$(lvdisplay -C -o "lv_path,lv_dm_path,lv_kernel_minor" | grep /dev/$VG_NAME/$LV_NAME | awk '{print $2}')
        lvMountPoint=$(df -h | grep "$lvDmPath" | awk '{print $NF}')

        if [[ -n $lvMountPoint ]]; then
            LOG "info" "Logical volume /dev/$VG_NAME/$LV_NAME is mounted on $lvMountPoint."
            PHISONAI_CACHE_PATH=$lvMountPoint
            update_env_cache_path
        else
            LOG "info" "Logical volume /dev/$VG_NAME/$LV_NAME is not a mount point."

            mountpoint -q -- "$PHISONAI_CACHE_PATH"
            isMountPoint=$?
            if [[ "$isMountPoint" -eq 0 ]]; then
                LOG "warning" "ENV-IST-D00009 Default path $PHISONAI_CACHE_PATH is already a mount point."
                return 1
            else
                LOG "info" "Mount Logical volume to default path $lvMountPoint."
                mount_lvm
                add_raid0_fstab
            fi
        fi

    ## If LVM not exist
    else
        LOG "info" "Logical volume /dev/$VG_NAME/$LV_NAME do not exist."
        if mountpoint -q -- "$PHISONAI_CACHE_PATH"; then
            if ! check_if_is_ai100; then
                LOG "warning" "ENV-IST-D00009 Default path $PHISONAI_CACHE_PATH is already a mount point."
                return 1
            fi
        else
            handle_disk
        fi
    fi

}


#### LVM not exist
function handle_disk() {
    mountpoint -q -- "$PHISONAI_CACHE_PATH"
    isMountPoint=$?
    if [[ "$isMountPoint" -eq 0 ]]; then
        LOG "warning" "Default path $PHISONAI_CACHE_PATH is already a mount point."
        return 1
    fi

    local availableDiskList
    get_available_disk availableDiskList
    LOG "info" "Available disk: $(echo "${availableDiskList[@]}" | sed 's/\(,\s*\)+/,/g'))"

    if [[ "$MODE" == "gui" ]]; then
        select_disk_by_gui
    elif [[ "$MODE" == "cli" ]]; then
        select_disk_by_cli
    fi
}


#### Select disk by GUI
function select_disk_by_gui() {
    while true; do
        selectedDisk=$(yad --list \
                            --multiple --checklist \
                            --print-column=3 \
                            --separator="," \
                            --width=600 --height=200 --center \
                            --title="Select disk" \
                            --text="Please select one or more disk to use as aiDAPTIVCache." \
                            --column="Pick" --column="Num" --column="Device" --column="Model" --column="Size" "${availableDiskList[@]}" \
                            --button="Next:0" )

        result=$?
        if [[ "$result" -eq 0 ]]; then
            selectedDisk=$(echo $selectedDisk | sed 's/\(,\s*\)+/,/g')
            LOG "info" "User select disk: $selectedDisk"
            IFS=',' read -r -a raid0_devices <<< "$selectedDisk"

            ## No disk selected
            if [[ -z $selectedDisk ]]; then
                msg="No disk selected, please select at least 1 disk."
                if [ "$MODE" == "gui" ]; then
                    yad_paned "info" "Empty selected" "$msg"
                fi
                LOG "info" "$msg"
            fi

            LOG "info" "PHISONAI_CACHE_DISK number of count is ${#raid0_devices[@]}."
            if [ ${#raid0_devices[@]} -gt 0 ]; then
                break
            fi
        elif [[ "$result" -eq 252 ]]; then
            exit 0
        fi
    done
}


#### Select disk by CLI
function select_disk_by_cli() {
    echo "Number disk:"
    for blkIndex in "${!unusedBlkDevices[@]}"; do
        blkDeviceModel=$(lsblk -P -o MODEL /dev/${unusedBlkDevices[$blkIndex]} | grep -v "\"\"" | awk -F\" '{print $2}')
        blkDeviceSize=$(lsblk -P -o SIZE /dev/${unusedBlkDevices[$blkIndex]} | grep -v "\"\"" | awk -F\" '{print $2}' | head -n 1)

        ## Filter only nvme
        nvmeFr=$(nvme id-ctrl /dev/${unusedBlkDevices[$blkIndex]} -o json | jq -r '.fr')
        for id in $(jq -r '.devices[].fw_id' $FIRMWARE_INFO_FILE); do
            if [[ "$nvmeFr" == "$id"* ]]; then
                echo "Devices $blkIndex: /dev/${unusedBlkDevices[${blkIndex}]}, Model: $blkDeviceModel, Size: $blkDeviceSize"
            fi
        done
    done

    echo
    read -p "Please select one or more disk to use as aiDAPTIVCache. (enter the index numbers separated by space): " selectedIndexs

    if [ -z "$selectedIndexs" ]; then
        LOG "error" "Device cannot be empty."
        return 1
    fi

    for index in $selectedIndexs; do
        if [ "$index" -ge ${#unusedBlkDevices[@]} ]; then
            LOG "error" "Invalid input. Exiting."
            return 1
        fi
        selectedIndexs+=($index)
        selectedDevices+=(${unusedBlkDevices[$index]})
    done

    ## 檢查是否有重複輸入的硬碟
    if [ ${#selectedDevices[@]} -ne 1 ]; then
        for (( i=0; i<${#selectedIndexs[@]}; i++ )); do
            for (( j=i+1; j<${#selectedIndexs[@]}; j++ )); do
                if [[ ${selectedIndexs[i]} == ${selectedIndexs[j]} ]]; then
                    LOG "error" "ENV-IST-D00012 Duplicate data entered ${selectedIndexs[i]}"
                    return 1
                fi
            done
        done
    fi

    for device in "${selectedDevices[@]}"; do
        raid0_devices+=(/dev/$device)
    done
}


#### Create and format single ai100 disk
function create_single_disk() {
    format_xfs ${raid0_devices[0]}

    check_device_type ${raid0_devices[0]}

    mountpoint -q -- "$PHISONAI_CACHE_PATH"
    isMountPoint=$?
    if [[ "$isMountPoint" -eq 0 ]]; then
        LOG "warning" "ENV-IST-D00009 Default path $PHISONAI_CACHE_PATH is already a mount point."
        return 1
    else
        mount ${raid0_devices[0]} "$PHISONAI_CACHE_PATH"
    fi

    if [ $? -eq 0 ]; then
        if grep -q "$PHISONAI_CACHE_PATH" /etc/fstab; then
            sed -i "\~$PHISONAI_CACHE_PATH~d" /etc/fstab
        fi
        UUID_NAME=$(blkid -s UUID | grep ${raid0_devices[0]} | awk -F'["]' '{print$ 2}')
        echo "UUID=$UUID_NAME $PHISONAI_CACHE_PATH xfs defaults,nofail 0 0" >> /etc/fstab
    else
        if [ "$MODE" == "gui" ]; then
            yad_paned "error" "Failed" "ENV-IST-D00010 Failed mount ${raid0_devices[0]} to $PHISONAI_CACHE_PATH."
        fi
        LOG "error" "ENV-IST-D00010 Failed mount ${raid0_devices[0]} to $PHISONAI_CACHE_PATH."
        return 1
    fi
}


#### Create and mount multiple ai100 disk
function create_raid0() {
    for (( i=0; i<${#raid0_devices[@]}; i++ )); do
        pv_create ${raid0_devices[$i]}
    done

    vg_create "${raid0_devices[@]}"
    lv_create

    format_xfs /dev/$VG_NAME/$LV_NAME

    check_device_type /dev/$VG_NAME/$LV_NAME

    ## Mount and add fstab
    mountpoint -q -- "$PHISONAI_CACHE_PATH"
    isMountPoint=$?
    if [[ "$isMountPoint" -eq 0 ]]; then
        LOG "warning" "ENV-IST-D00009 Default path $PHISONAI_CACHE_PATH is already a mount point."
        return 1
    else
        mount_lvm
    fi

    add_raid0_fstab
}


function remove_raid0() {
    umount_path "$PHISONAI_CACHE_PATH"

    lvdisplay /dev/$VG_NAME/$LV_NAME > /dev/null 2>&1
    result=$?
    if [ "$result" -eq 0 ]; then
        lv_remove
    fi

    vgdisplay "$VG_NAME" > /dev/null 2>&1
    result=$?
    if [ "$result" -eq 0 ]; then
        pv_list=$(pvs --separator , --no-heading -o pv_name --select vg_name=$VG_NAME | awk -F, '{print $1}' | tr "," "\n")
        vg_remove
    fi

    for pv in $(echo "$pv_list" | tr "," "\n")
    do
        pv_remove_result=$(pvremove -y -f "$pv")
        if [ $? -ne 0 ]; then
            if [ "$MODE" == "gui" ]; then
                yad_paned "error" "Install Finished" "ENV-IST-D00003 Failed remove physical volume $pv, $pv_remove_result."
            fi
            echo "error" "ENV-IST-D00003 Failed remove physical volume $pv, $pv_remove_result."
        fi
    done
}


#### Fstab ####
function add_raid0_fstab() {
    if grep -q "$PHISONAI_CACHE_PATH" /etc/fstab; then
        sed -i "\~$PHISONAI_CACHE_PATH~d" /etc/fstab
    fi

    lv_dmpath=$(lvdisplay -C -o "lv_path,lv_dm_path,lv_kernel_minor" | grep /dev/$VG_NAME/$LV_NAME | awk '{print $2}')
    lv_uuid=$(blkid -s UUID | grep $lv_dmpath | grep -v "rimage" | awk -F'["]' '{print $2}')
    echo "UUID=$lv_uuid $PHISONAI_CACHE_PATH xfs defaults,nofail 0 0" >> /etc/fstab
}


function remove_from_fstab() {
    if grep -q "$PHISONAI_CACHE_PATH" /etc/fstab; then
        LOG "info" "Remove $PHISONAI_CACHE_PATH from /etc/fstab"
        sed -i "\~$PHISONAI_CACHE_PATH~d" /etc/fstab
    fi
}


function format_xfs() {
    device=$1

    yes | mkfs.xfs -f "$device" 2>&1 | while IFS= read -r line; do
        LOG "info" "$line"
    done

    result=$?
    if [[ "$result" -ne 0 ]]; then
        msg="ENV-IST-D00001 Failed format $device to xfs."
        if [[ "$MODE" == "gui" ]]; then
            yad_paned "error" "Failed" "$msg"
        fi
        LOG "error" "$msg"
        return 1
    fi
}

function check_if_is_ai100() {
    block_device=$(df --output=source $PHISONAI_CACHE_PATH | awk 'NR==2 {print $1}')
    fw_version=$(nvme id-ctrl $block_device -o json | jq '.fr')
    if [[ $fw_version == *"EIFZ"* ]]; then
        return 0
    else
        return 1
    fi
}


function check_device_type() {
    device=$1
    LOG "info" "Device type: $(file -sL $device)"
}


function mount_lvm() {
    lvchange -ay /dev/prosuite-vg/prosuite-rd

    mount /dev/"$VG_NAME"/"$LV_NAME" "$PHISONAI_CACHE_PATH" 2>&1
    result=$?
    if [[ "$result" -ne 0 ]]; then
        msg="ENV-IST-D00010 Failed to mount /dev/$VG_NAME/$LV_NAME to $PHISONAI_CACHE_PATH."
        if [[ "$MODE" == "gui" ]]; then
            yad_paned "error" "Failed" "$msg"
        fi
        LOG "error" "$msg"
        return 1
    fi
}


function umount_path() {
    UMOUNT_PATH=$1
    umount "$UMOUNT_PATH" 2>&1 | while IFS= read -r line; do
        LOG "info" "$line"
    done

    result=$?
    if [[ "$result" -ne 0 ]]; then
        msg="ENV-IST-D00011 Failed umount $UMOUNT_PATH."
        if [[ "$MODE" == "gui" ]]; then
            yad_paned "error" "Failed" "$msg"
        fi
        LOG "error" "$msg"
    fi
}


#### PV, VG, LV ####
function pv_create() {
    blk_device=$1

    pvdisplay "$blk_device" > /dev/null 2>&1
    result=$?
    if [[ "$result" -eq 0 ]]; then
        :
    else
        LOG "info" "Wipefs disk $blk_device."

        wipefs --all "$blk_device" 2>&1 | while IFS= read -r line; do
            LOG "info" "$line"
        done

        result=$?
        if [[ "$result" -ne 0 ]]; then
            msg="ENV-IST-D00008 Failed wipefs disk $blk_device."
            if [ "$MODE" == "gui" ]; then
                yad_paned "warning" "Warning" "$msg"
            fi
            LOG "warning" "$msg"
        fi

        pv_create_result=$(pvcreate "$blk_device" 2>&1)
        result=$?
        if [[ "$result" -eq 0 ]]; then
            pvdisplay "$blk_device" 2>&1 | while IFS= read -r line; do
                LOG "info" "$line"
            done
        else
            if [ "$MODE" == "gui" ]; then
                yad_paned "error" "Failed" "ENV-IST-D00004 Failed create physical volume $blk_device\n$pv_create_result."
            fi
            LOG "ENV-IST-D00002 Failed create physical volume $blk_device, $pv_create_result."
            return 1
        fi
    fi
}


function lv_create() {
    LOG "info" "Create logical volume /dev/$VG_NAME/$LV_NAME."
    lv_create_result=$(lvcreate --type=raid0 -l100%FREE -n $LV_NAME $VG_NAME 2>&1)
    result=$?
    if [[ "$result" -eq 0 ]]; then
        lvdisplay /dev/"$VG_NAME"/"$LV_NAME" 2>&1 | while IFS= read -r line; do
            LOG "info" "$line"
        done
    else
        msg="ENV-IST-D00006 Failed create logical volume $LV_NAME, $lv_create_result."
        if [ "$MODE" == "gui" ]; then
            yad_paned "error" "Failed" "$msg"
        fi
        LOG "error" "$msg"
        return 1
    fi
}

function lv_remove() {
    LOG "info" "Remove logical volume /dev/$VG_NAME/$LV_NAME."
    lv_remove_result=$(lvremove -y -f /dev/$VG_NAME/$LV_NAME 2>&1)
    result=$?
    if [[ "$result" -ne 0 ]]; then
        msg="ENV-IST-D00007 Failed remove logical volume /dev/$VG_NAME/$LV_NAME, $lv_remove_result"
        if [ "$MODE" == "gui" ]; then
            yad_paned "error" "Failed" "$msg"
        fi
        LOG "error" "$msg"
    fi
}


function vg_create() {
    blk_devices=("$@")

    vgdisplay $VG_NAME > /dev/null 2>&1
    result=$?
    if [[ "$result" -eq 0 ]]; then
        vg_remove
    fi

    vg_create_result=$(vgcreate "$VG_NAME" ${blk_devices[@]} 2>&1)
    result=$?
    if [[ "$result" -eq 0 ]]; then
        vgdisplay "$VG_NAME" 2>&1 | while IFS= read -r line; do
            LOG "info" "$line"
        done
    else
        msg="ENV-IST-D00004 Failed create volume group, $vg_create_result."
        if [[ "$MODE" == "gui" ]]; then
            yad_paned "error" "Failed" "$msg"
        fi
        LOG "error" "$msg"
        return 1
    fi
}


function vg_remove() {
    vg_remove_result=$(vgremove -y -f $VG_NAME 2>&1)
    result=$?
    if [[ "$result" -ne 0 ]]; then
        msg="ENV-IST-D00005 Failed remove volume group $VG_NAME, $vg_remove_result."
        if [[ "$MODE" == "gui" ]]; then
            yad_paned "error" "Failed" "$msg"
        fi
        LOG "error" "$msg"
        exit 1
    fi
}

#### Download from huggingface
function download_llm_model() {
    llmModel=$1

    LOG "info" "Download LLM model $llmModel to $DEFAULT_MODEL_DST/$llmModel."
    git clone https://"$HUGGINGFACE_USER":"$HUGGINGFACE_TOKEN"@huggingface.co/meta-llama/"$llmModel" "$DEFAULT_MODEL_DST"/"$llmModel"
    result=$?
    if [[ "$result" -ne 0 ]]; then
        msg="ENV-IST-M00001 Download LLM model $llmModel failed, remove folder $DEFAULT_MODEL_DST/$llmModel."
        if [ "$MODE" == "gui" ]; then
            yad_paned "warning" "Failed" "$msg"
        fi
        LOG "warning" "$msg"
        LOG "info" "Remove failed download folder $DEFAULT_MODEL_DST/$llmModel."
        rm -rf "$DEFAULT_MODEL_DST"/"$llmModel"
    else
        LOG "info" "Download LLM model $llmModel success, remove .git folder."
        rm -rf "$DEFAULT_MODEL_DST"/"$llmModel"/.git
    fi
}


#### Normal install
#### Download service model
function download_model_gui() {
    while true; do
        selectModel=$(yad --list \
                        --title="Select model to download" \
                        --print-column=1 \
                        --separator="" \
                        --width=700 --height=400 --center \
                        --column="Num" --column="Model Name" \
                        1 "Llama-2-7b-chat-hf" \
                        2 "Meta-Llama-3-8B-Instruct" \
                        --button="Previous:1" \
                        --button="Download:0" )

        selectState=$?
        if [[ "$selectState" -eq 0 ]]; then
            case "$selectModel" in
                1)
                    download_llm_model "Llama-2-7b-chat-hf"
                ;;
                2)
                    download_llm_model "Meta-Llama-3-8B-Instruct"
                ;;
            esac
        else
            return 1
        fi
    done
}


#### Normal install
#### Download service model
function download_model_cli() {
    echo "Select model to download:"
    echo "0. Cancel"
    echo "1. Llama-2-7b-chat-hf"
    echo "2. Meta-Llama-3-8B-Instruct"

    read -p "Model No.:" selectModel
    case "$selectModel" in
    0)
        echo "Download canceled"
        ;;
    1)
        download_llm_model "Llama-2-7b-chat-hf"
        ;;
    2)
        download_llm_model "Meta-Llama-3-8B-Instruct"
        ;;
    *)
        echo "Please input number 1-2"
        ;;
    esac
}


#### Normal install
function get_km_api() {
    KM_API_FOLDER="$PHISONAI_DATA_SRC_PATH/service-model/km-api/multilingual-e5-large"
    if [[ ! -d "$KM_API_FOLDER" ]]; then
        LOG "info" "Git clone km-api to $KM_API_FOLDER."
        git clone --progress --verbose https://huggingface.co/intfloat/multilingual-e5-large "$KM_API_FOLDER"

        result=$?
        if [[ "$result" -eq 0 ]];then
            rm -rf "$KM_API_FOLDER"/.git
        else
            rm -rf "$KM_API_FOLDER"
            if [[ "$MODE" == "gui" ]]; then
                yad_paned "error" "Failed" "ENV-IST-M00001 Download km-api intfloat/multilingual-e5-large failed."
            fi
            LOG "error" "ENV-IST-M00001 Download km-api intfloat/multilingual-e5-large failed."
            exit 1
        fi
    fi
}


#### Other ####
function clean_memory_cache() {
    origin_cache_size=$(free -h | awk '/^Mem:/ {print $6}')

    echo 1 > /proc/sys/vm/drop_caches
    result=$?
    if [[ "$result" -eq 0 ]]; then
        new_cache_size=$(free -h | awk '/^Mem:/ {print $6}')
        msg1="The original amount of memory cache used is $origin_cache_size bytes."
        msg2="After clean, memory cache used is $new_cache_size."
        if [[ "$MODE" == "gui" ]]; then
            yad_paned "info" "Release memory cache" "$msg1\n$msg2"
        fi
        LOG "info" "$msg1, $msg2"
    fi
}


function replace_db_data_path() {
    if [[ -d "$PG_DATA_PATH/data/data" ]]; then
        mv "$PG_DATA_PATH"/data/data/* "$PG_DATA_PATH"/data > /dev/null 2>&1
        result=$?
        if [[ "$result" -eq 0 ]]; then
            rm -rf "$PG_DATA_PATH"/data/data
        fi
        chown 1001:1001 -R "$PG_DATA_PATH"
    fi
}


#### Uninstall ####
function uninstall() {  
    rm_lvm=FALSE
    umount_single_disk=FALSE

    lvdisplay /dev/"$VG_NAME"/"$LV_NAME" > /dev/null 2>&1
    result=$?
    if [ "$result" -eq 0 ]; then
        LOG "info" "Existing through LVM composed of install.sh."
        aiDAPTIV_CACHE="LVM"
    elif [[ $(df -P "$PHISONAI_CACHE_PATH" | tail -n 1 | awk '{print $6}') != "/" ]]; then
        parent_dir=$(df -P "$PHISONAI_CACHE_PATH" | tail -n 1 | awk '{print $1}')
        result=$(ls "$parent_dir" | grep -q "/dev/nvme")
        if [[ "$result" -eq 0 ]]; then
            LOG "info" "ai100 mounted through install.sh exists."
            aiDAPTIV_CACHE="DISK"
        fi
    fi

    ## Ask if remove LVM raid
    ## Only occure when LVM is combine by install.sh
    if [[ "$aiDAPTIV_CACHE" == "LVM" ]]; then
        msg="Do you want to remove aiDAPTIV+ cache LVM RAID0 ?"
        if [[ "$MODE" == "gui" ]]; then
            yad --question \
                --width=400 --height=100 --center \
                --title="Remove aiDAPTIV+ cache LVM RAID0" \
                --text="$msg" \
                --button="Cancel:1" \
                --button="No:2" \
                --button="Yes:0"

            result=$?
        elif [[ "$MODE" == "cli" ]]; then
            result=$(ask_question "$msg")
        fi

        if [[ "$result" -eq 0 ]]; then
            rm_lvm=TRUE
            PHISONAI_CACHE_PATH=$(df -P /dev/$VG_NAME/$LV_NAME | awk 'NR!=1{print $6}')
        elif [[ "$result" -eq 2 ]]; then
            rm_lvm=FALSE
        else
            return 1
        fi
    fi
    LOG "info" "Remove LVM: $rm_lvm."

    ## Ask if umount single disk
    if [[ "$aiDAPTIV_CACHE" == "DISK" ]]; then
        msg="Do you want to umount $PHISONAI_CACHE_PATH that aiDAPTIV+ cache used ?"
        if [[ "$MODE" == "gui" ]]; then
            yad --question \
                --width=400 --height=100 --center \
                --title="Single aiDAPTIV+ cache disk" \
                --text="$msg" \
                --button="Cancel:1" \
                --button="No:2" \
                --button="Yes:0"
            result=$?
        elif [[ "$MODE" == "cli" ]]; then
            result=$(ask_question "$msg")
        fi
        if [[ "$result" -eq 0 ]]; then
            umount_single_disk=TRUE
        elif [[ "$result" -eq 2 ]]; then
            umount_single_disk=FALSE
        else
            return 1
        fi
    fi
    LOG "info" "Remove ai100 single disk: $umount_single_disk."

    if command -v docker &> /dev/null; then
        ## Ask if remove docker image
        result=$(ask_question "Do you want to remove Pro Suite docker image ?")

        if [[ "$result" -eq 0 ]]; then
            rm_image=TRUE
        elif [[ "$result" -eq 2 ]]; then
            rm_image=FALSE
        else
            exit 1
        fi
        log_info "Remove docker image that Pro Suite used: $rm_image."
    fi

    if [[ "$result" -eq 0 ]]; then
        rm_image=TRUE
    elif [[ "$result" -eq 2 ]]; then
        rm_image=FALSE
    else
        return 1
    fi
    LOG "info" "Remove docker image that Pro Suite used: $rm_image."

    ## Ask if remove prosuite date
    ## Remove /mnt/data/local-data/*
    ## Remove /opt/phisonai/data/*
    msg="Do you want to remove Pro Suite data ?"
    if [[ "$MODE" == "gui" ]]; then
        yad --question \
            --width=400 --height=100 --center \
            --title="Remove image" \
            --text="$msg" \
            --button="Cancel:1" \
            --button="No:2" \
            --button="Yes:0"

        result=$?
    elif [[ "$MODE" == "cli" ]]; then
        result=$(ask_question "$msg")
    fi

    if [[ "$result" -eq 0 ]]; then
        rm_data=TRUE
    elif [[ "$result" -eq 2 ]]; then
        rm_data=FALSE
    else
        return 1
    fi
    LOG "info" "Remove data that Pro Suite used: $rm_data."

    ## Start uninstall
    if command -v docker &> /dev/null; then
        if [ "$rm_image" == "TRUE" ]; then
            dc_stop "rmi"
        else
            dc_stop
        fi
    fi

    if [ "$rm_data" == "TRUE" ]; then
        remove_prosuite_service_data
    fi

    if [ "$rm_lvm" == "TRUE" ]; then
        remove_raid0
        remove_from_fstab
    elif [ "$umount_single_disk" == "TRUE" ]; then
        umount_path "$PHISONAI_CACHE_PATH"
        remove_from_fstab
    fi

    if command -v docker &> /dev/null; then
        remove_docker_network
    fi
    
    remove_apt_sourcelist
    LOG "info" "Finish uninstall Pro Suite."

    crontab -u root -l | grep "@reboot docker compose" | wc -l
    if [[ $? -eq 0 ]]; then
        crontab -u root -l | grep -v "\@reboot docker compose" | crontab -u root -
    fi
}


#### Debug ####
function debug_mode_gui() {
    debug_action=$(yad --list \
                --title="Select an option" \
                --width=700 --height=400 --center \
                --print-column=1 \
                --separator="" \
                --column="Num" --column="Action" --column="Describe" \
                1 "Restart specific service" "Restart Pro Suite specific service" \
                2 "Pull docker image" "Pull phison-image.env all image" \
                3 "Cleanup memory buffer/cache" "Cleanup memory buffer/cache" \
                4 "Mount ai100" "Create LVM or mount disk directly"\
                5 "Update Embedded Model" "Update Embedded Model"\
                --button="Previous:2" \
                --button="Confirm:0" )

    result=$?
    if [[ "$result" -eq 0 ]]; then
        do_debug_mode "$debug_action"
    else
        return 1
    fi
}


function debug_mode_cli() {
    echo "Select an option:"
    printf "%-40s %-40s\n" " 1) Restart specific service" "Restart Pro Suite specific service"
    printf "%-40s %-40s\n" " 2) Pull docker image" "Pull phison-image.env all image"
    printf "%-40s %-40s\n" " 3) Cleanup memory buffer/cache" "Cleanup memory buffer/cache"
    printf "%-40s %-40s\n" " 4) Mount ai100" "Create LVM or mount disk directly"
    printf "%-40s %-40s\n" " 5) Update Embedded Model" "Update Embedded Model"

    read -p "Option: " option
    until [[ "$option" =~ ^[1-5]$ ]]; do
        echo "$option: invalid selection."
        read -p "Option: " option
    done

    do_debug_mode "$option"
}


function do_debug_mode() {
    option=$1
    case "$option" in
        1)
            dc_restart_specific
        ;;
        2)
            get_docker_image
        ;;
        3)
            clean_memory_cache
        ;;
        4)
            check_lvm
            if [ ${#raid0_devices[@]} -eq 1 ]; then
                create_single_disk
            elif [ ${#raid0_devices[@]} -gt 1 ]; then
                create_raid0
            fi
            if [[ $? -ne 0 ]]; then
                exit 1
            fi
        ;;
        5)
            get_km_api
            if [[ $? -ne 0 ]]; then
                exit 1
            fi
            update_embedded_model
            if [[ $? -ne 0 ]]; then
                if [[ "$MODE" == "gui" ]]; then
                    yad_paned "warning" "Update embedded model failed" "Failed to update embedded model"
                fi
                LOG "warning" "Update embedded model failed"
            fi
       
    esac
}


function ask_question() {
    local question="$1"
    local result

    read -p "$question (Yes/No/Cancel): " answer
    answer=$(echo "$answer" | tr '[:upper:]' '[:lower:]')
    if [ "$answer" = "yes" ] || [ "$answer" = "y" ]; then
        result=0
    elif [ "$answer" = "no" ] || [ "$answer" = "n" ]; then
        result=2
    elif [ "$answer" = "cancel" ]; then
        result=1
    else
        echo "(Yes/No/Cancel)。"
        result=$(ask_question "$question")
    fi

    echo "$result"
}


function update_embedded_model() {
    LOG "info" "Converting embedding model..."

    ans=1
    dc_stop
    c_name=$($DOCKER_CLI config  --services | grep "mongo\|chroma\|km-api"  |  tr '\n' ' ' )
    sed -i 's/#EMBEDING_DEVICE=cuda:0/EMBEDING_DEVICE=cuda:0/' $DOCKER_CONFIG_ENV 
    sed -i 's/KM_GPU_COUNT=0/KM_GPU_COUNT=1/' $DOCKER_CONFIG_ENV 
    $DOCKER_CLI up -d $c_name
    sed -i 's/EMBEDING_DEVICE=cuda:0/#EMBEDING_DEVICE=cuda:0/' $DOCKER_CONFIG_ENV 
    sed -i 's/KM_GPU_COUNT=1/KM_GPU_COUNT=0/' $DOCKER_CONFIG_ENV 
    cnt=1
    while true
    do
        res=$(curl -s -X 'POST' 'http://localhost:3019/database/rag/collection/update_chroma_by_mongodb'  -H 'accept: application/json'  -H 'Content-Type: application/json' -d '{}')
        status=$(echo $res | jq -r '.action_status')
        if [ ! -z "$status"  ] && [ $status == 1 ]; then
            ans=0
            break
        fi
        if [ $cnt -gt 10 ] ;  then
            break
        fi
        echo 'Retry sleep 5s'
        sleep 5s
        (( cnt++ ))
    done
    dc_stop
    return $ans
}


function main_gui() {
    check_root
    check_log_exist
    check_yad
    check_secure_boot
    check_bios_mode

    update_memsize
    update_timezone

    replace_db_data_path

    while true; do
        selectAction=$(yad --list \
                --title="Select an action" \
                --width=800 --height=400 --center \
                --print-column=1 \
                --separator="" \
                --column="Num" --column="Action" --column="Describe" \
                0 "Install Pro Suite" "Mount ai100 disk, apt install" \
                1 "Get Pro Suite status" "Get Pro Suite container status" \
                2 "Upgrade Pro Suite all services" "Upgrade Pro Suite to current installation package version" \
                3 "Start Pro Suite all services" "Startup Pro Suite service" \
                4 "Stop Pro Suite all services" "Stop Pro Suite service" \
                5 "Restart Pro Suite all services" "Only restart Pro Suite service" \
                6 "Uninstall Pro Suite all services" "Stop and Uninstall Pro Suite services" \
                7 "Enter Debug Mode " "Debug mode" \
                8 "Exit Script" "Exit" \
                --button="Exit:2" \
                --button="Select:0" )


        selectState=$?
        if [[ $selectState -eq 1 ]]; then
            msg1="X11 forwarding needs to be enabled (using the -X option) on the SSH connection to use the graphical interface."
            LOG "warning" "$msg1"
            exit 1
        elif [[ $selectState -eq 2 || $selectState -eq 252 ]]; then
            LOG "info" "Exit"
            return 0
        fi

        case "$selectAction" in
            0)
                init_directory > /dev/null 2>&1

                ## Stop Pro Suite container if docker cli exist
                if command -v docker &> /dev/null; then
                    LOG "info" "Docker cli exist, stop Pro Suite container."
                    dc_stop
                fi

                ## APT install
                apt_update
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi

		disable_auto_upgrade
		if [[ $? -ne 0 ]]; then
                    exit 1
                fi

                apt_before_install
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi

		update_firmware_startup
		if [[ $? -ne 0 ]]; then
                    exit 1
                fi

                add_docker_apt_key
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi
                
                add_nvidia_apt_key
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi

                ## Check if LVM exist
                check_lvm
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi

                ## APT install
                apt_update
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi

                apt_after_install
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi

                ## Docker config
                config_docker
                start_docker_service

                create_docker_network 
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi

                get_docker_image

                ## Embeding model
                get_km_api
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi

                ## Mount ai100
                if [ ${#raid0_devices[@]} -eq 1 ]; then
                    create_single_disk
                elif [ ${#raid0_devices[@]} -gt 1 ]; then
                    create_raid0
                fi
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi

                ## Check reboot service
                check_reboot
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi

                update_embedded_model
                if [[ $? -ne 0 ]]; then
                    yad_paned "warning" "Update embedded model failed" "Failed to update embedded model"
                fi

                dc_startup
                if [[ $? -ne 0 ]]; then
                    yad_paned "error" "Failed start" "failed startup Pro Suite service."
                fi
            ;;
            1)
                dc_status
            ;;
            2)
                init_directory > /dev/null 2>&1
		disable_auto_upgrade
		if [[ $? -ne 0 ]]; then
                    exit 1
                fi

                update_firmware_startup
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi

                create_docker_network
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi

                check_lvm
                if [[ $? -eq 0 ]];then
                    dc_stop
                    get_docker_image

                    get_km_api
                    if [[ $? -ne 0 ]]; then
                        exit 1
                    fi
                    update_embedded_model
                    if [[ $? -ne 0 ]]; then
                        yad_paned "warning" "Update embedded model failed" "Failed to update embedded model"
                    fi
                    dc_startup
                    if [[ $? -ne 0 ]]; then
                        yad_paned "error" "Failed start" "failed startup Pro Suite service."
                    fi
                fi
            ;;
            3)
                init_directory > /dev/null 2>&1
                create_docker_network
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi

                dc_startup
                if [[ $? -ne 0 ]]; then
                    yad_paned "error" "Failed start" "failed startup Pro Suite service."
                fi
            ;;
            4)
                dc_stop
            ;;
            5)
                init_directory > /dev/null 2>&1
                create_docker_network
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi

                dc_stop
                sleep 5
                dc_startup
                if [[ $? -ne 0 ]]; then
                    yad_paned "error" "Failed start" "failed startup Pro Suite service."
                fi
            ;;
            6)
                apt_update
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi

                apt_before_install
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi
                uninstall
            ;;
            7)
                debug_mode_gui
            ;;
            8)
                exit 0
        esac
    done
}


function main_cli() {
    check_root
    check_log_exist
    check_secure_boot
    check_bios_mode

    update_memsize
    update_timezone

    echo "Select an option:"
    printf "%-40s %-40s\n" " 0) Install Pro Suite" "Mount ai100 disk, apt install."
    printf "%-40s %-40s\n" " 1) Get Pro Suite status" "Get Pro Suite container status."
    printf "%-40s %-40s\n" " 2) Upgrade Pro Suite all services" "Upgrade Pro Suite to the previous installation package version."
    printf "%-40s %-40s\n" " 3) Start Pro Suite all services" "Startup Pro Suite service."
    printf "%-40s %-40s\n" " 4) Stop Pro Suite all services" "Stop Pro Suite service."
    printf "%-40s %-40s\n" " 5) Restart Pro Suite all services" "Only restart Pro Suite service."
    printf "%-40s %-40s\n" " 6) Uninstall Pro Suite all services" "Stop and Uninstall Pro Suite services."
    printf "%-40s %-40s\n" " 7) Enter Debug Mode" "Debug mode."
    printf "%-40s %-40s\n" " 8) Exit Script" "Exit."

    read -p "Option: " selectAction
    until [[ "$selectAction" =~ ^[0-9]$|10$ ]]; do
        LOG "warning" "$selectAction: invalid selection."
        read -p "Option: " selectAction
    done

    echo
    case "$selectAction" in
        0)
            init_directory > /dev/null 2>&1
            ## Stop Pro Suite container if docker cli exist
            if command -v docker &> /dev/null; then
                LOG "info" "Docker cli exist, stop Pro Suite container."
                dc_stop
            fi

            ## APT install
            apt_update
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            disable_auto_upgrade
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            apt_before_install
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            update_firmware_startup
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            add_docker_apt_key
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            add_nvidia_apt_key
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            ## Check if LVM exist
            check_lvm
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            ## APT install
            apt_update
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            apt_after_install
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            ## Docker config
            config_docker
            start_docker_service
            create_docker_network
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            get_docker_image

            ## Embeding model
            get_km_api
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            ## Mount ai100
            if [ ${#raid0_devices[@]} -eq 1 ]; then
                create_single_disk
            elif [ ${#raid0_devices[@]} -gt 1 ]; then
                create_raid0
            fi
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            ## Check reboot
            check_reboot
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            #update_embedded_model
            #if [[ $? -ne 0 ]]; then
            #    LOG "warning" "Update embedded model failed"
            #fi

            dc_startup
            if [[ $? -ne 0 ]]; then
                LOG "error" "failed startup Pro Suite service."
            fi
        ;;
        1)
            dc_status
        ;;
        2)
            init_directory > /dev/null 2>&1
	    disable_auto_upgrade
	    if [[ $? -ne 0 ]]; then
                exit 1
            fi

	    update_firmware_startup
	    if [[ $? -ne 0 ]]; then
                exit 1
            fi

            create_docker_network
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            check_lvm
            if [[ $? -eq 0 ]];then
                dc_stop
                get_docker_image

                get_km_api
                if [[ $? -ne 0 ]]; then
                    exit 1
                fi
                update_embedded_model
                if [[ $? -ne 0 ]]; then
                    LOG "warning" "Update embedded model failed"
                fi
                dc_startup
                if [[ $? -ne 0 ]]; then
                    LOG "error" "failed startup Pro Suite service."
                fi
            fi
        ;;
        3)
            init_directory > /dev/null 2>&1
            create_docker_network
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            dc_startup
            if [[ $? -ne 0 ]]; then
                LOG "error" "failed startup Pro Suite service."
            fi
        ;;
        4)
            dc_stop
        ;;
        5)
            init_directory > /dev/null 2>&1
            create_docker_network
            if [[ $? -ne 0 ]]; then
                exit 1
            fi
            
            dc_stop
            sleep 5
            dc_startup
            if [[ $? -ne 0 ]]; then
                LOG "error" "failed startup Pro Suite service."
            fi
        ;;
        6)
            apt_update
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            apt_before_install
            if [[ $? -ne 0 ]]; then
                exit 1
            fi

            uninstall
        ;;
        7)
            debug_mode_cli
        ;;
        8)
            exit 0
    esac
}


function error_mode() {
    invalidValue=$1
    echo -e "$(date +%F\ %T) : [error] Invalid parameter '$invalidValue'"
    echo -e "$(date +%F\ %T) : [error] GUI mode (default) 'sudo bash install.sh'"
    echo -e "$(date +%F\ %T) : [error] CLI mode (without X11) 'sudo bash install.sh -c'"
    exit 1
}

#### Determine the mode used for installation
MODE="gui"
if [[ $# -eq 0 ]]; then
    :
elif [[ $# -eq 1 ]]; then
    if [[ "$1" == "-c" ]]; then
        MODE="cli"
    else
        error_mode $*
    fi
else
    error_mode $*
fi

#### Start main
echo -e "$(date +%F\ %T) : [info]"
echo -e "$(date +%F\ %T) : [info] Pro Suite install in $MODE mode."
echo -e "$(date +%F\ %T) : [info]"
if [[ "$MODE" == "gui" ]]; then
    main_gui
else
    main_cli
fi
